using System;
using System.Data;
using System.Reflection;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitAntigen;

namespace gov.va.med.vbecs.BOL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/17/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Blood Unit Antigen business object layer class
		///</summary>

		#endregion

	public class BloodUnitAntigen: BaseBusinessObject, IRecordStatusCode
	{
		#region Variables

		private Guid _bloodUnitGUID;
		private int _antigenTypeID;
		private bool _tested;
		private bool _positive;
		private Common.RecordStatusCode _recordStatusCode;
		private string _antigenName;
		

		#endregion

		#region Constructors
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="516"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Empty blood unit antigen object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5214"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blood Unit Antigen class constructor
		/// </summary>
		public BloodUnitAntigen()
		{
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2396"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>BloodUnitAntigen object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2478"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates BloodUnitAntigen object and populates it with data from DataRow
		/// </summary>
		public BloodUnitAntigen(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="515"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BloodUnitGuid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="668"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets blood unit uniques identifier
		/// </summary>
		public Guid BloodUnitGuid
		{
			get
			{
				return this._bloodUnitGUID;
			}
			set
			{
				this._bloodUnitGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="517"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="683"> 
		///		<ExpectedInput>Invalid antigen type ID (-1)</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="684"> 
		///		<ExpectedInput>Invalid antigen type ID (10000)</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the antigen type id of blood unit antigen
		/// </summary>
		public int AntigenTypeId
		{
			get
			{
				return this._antigenTypeID;
			}
			set
			{
				if ((value<0 || value >9999))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											value).ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				this._antigenTypeID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3610"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3611"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Tested?
		/// </summary>
		public bool Tested
		{
			get
			{
				return this._tested;
			}
			set
			{
				this._tested = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="518"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="670"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value to indicate if an antigen tested Positive
		/// </summary>
		public bool Positive
		{
			get
			{
				return this._positive;
			}
			set
			{
				this._positive = value;
			}
		}
	
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5215"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>stringg</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5216"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5217"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5218"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AntigenName
		/// </summary>
		public string AntigenName
		{
			get
			{
				return _antigenName;
			}
			set
			{
				_antigenName = value;
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="521"> 
		///		<ExpectedInput>Blood unit unique identifier and division code</ExpectedInput>
		///		<ExpectedOutput>Data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2295"> 
		///		<ExpectedInput>Invalid Guid/divisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Call data access layer to return table of antigens for given blood unit, CR 2899 
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns>Data table identifying antigens</returns>
		public static DataTable GetBloodUnitAntigens(Guid bloodUnitGuid, string divisionCode)
		{
			return BOL.BloodUnitAntigen.GetBloodUnitAntigens(bloodUnitGuid.ToString(), divisionCode);
		}
		
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2010</CreationDate>
		///
		///<TestCases>
		///
		///<Case type="0" testid ="9151"> 
		///		<ExpectedInput>Blood unit unique identifier string and division code</ExpectedInput>
		///		<ExpectedOutput>Data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9152"> 
		///		<ExpectedInput>Empty string/invalid divisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///	
		///	</TestCases>
		///
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Call data access layer to return table of antigens for given blood unit, CR 2899 
		/// </summary>
		/// <param name="bloodUnitGuidList"></param>
		/// <param name="divisionCode"></param>
		/// <returns>Data table identifying antigens</returns>
		public static DataTable GetBloodUnitAntigens(string bloodUnitGuidList, string divisionCode)
		{
			return DAL.BloodUnitAntigen.GetBloodUnitAntigens(bloodUnitGuidList,divisionCode);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5219"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5220"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetEmptyBloodUnitAntigenTable
		/// </summary>
		/// <param name="isUpdate">Add RowVersion</param>
		/// <returns>Data table</returns>
		public static System.Data.DataTable GetEmptyBloodUnitAntigenTable(bool isUpdate)
		{
			System.Data.DataTable dt = new System.Data.DataTable(Common.VbecsTables.BloodUnitAntigen.TableName);
			
			dt.Columns.Add(TABLE.BloodUnitGuid, typeof(System.Guid));
			dt.Columns.Add(TABLE.AntigenTypeId, typeof(System.Int32));
			dt.Columns.Add(TABLE.Positive, typeof(bool));
			dt.Columns.Add(TABLE.DivisionCode, typeof(string));
			dt.Columns.Add(TABLE.RecordStatusCode,typeof(char));
			dt.Columns.Add(TABLE.LastUpdateUser,typeof(string));
			dt.Columns.Add(TABLE.LastUpdateFunctionId, typeof(int));

			if (isUpdate)
				dt.Columns.Add(TABLE.RowVersion, typeof(Byte[]));

			DataColumn[] keys = {dt.Columns[TABLE.AntigenTypeId]};
			dt.PrimaryKey = keys;

			return dt;
		}
		
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5221"> 
		///		<ExpectedInput>BloodUnitGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5222"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>Empty Arraylist</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitAntigensList
		/// </summary>
		/// <param name="bloodUnitGuid">_bloodUnitGUID</param>
		/// <param name="divisionCode">_divisionCode</param>
		/// <returns>Array List</returns>
		public static System.Collections.ArrayList GetBloodUnitAntigensList(Guid bloodUnitGuid, string divisionCode)
		{
			System.Collections.ArrayList al = new System.Collections.ArrayList();
			System.Data.DataTable dt = DAL.BloodUnitAntigen.GetBloodUnitAntigens(bloodUnitGuid.ToString(), divisionCode);
			foreach(System.Data.DataRow dtRow in dt.Rows)
			{
				BOL.BloodUnitAntigen bloodUnitAntigen = new BOL.BloodUnitAntigen(dtRow);
				al.Add(bloodUnitAntigen);
			}
			return al;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/21/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5017"> 
		///		<ExpectedInput>BloodUnitGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5018"> 
		///		<ExpectedInput>Invalid BloodUNitguid, divisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retieves active and inactive antigens for a given BloodUnit
		/// </summary>
		/// <param name="bloodUnitGuid">_bloodUnitGUID</param>
		/// <param name="divisionCode">_divisionCode</param>
		/// <returns>Array List</returns>
		public static System.Collections.ArrayList GetAllAntigensForBloodUnitList(Guid bloodUnitGuid, string divisionCode)
		{
			System.Collections.ArrayList al = new System.Collections.ArrayList();
			System.Data.DataTable dt = DAL.BloodUnitAntigen.GetAllAntigensForBloodUnit(bloodUnitGuid, divisionCode);
			foreach(System.Data.DataRow dtRow in dt.Rows)
			{
				BOL.BloodUnitAntigen bloodUnitAntigen = new BOL.BloodUnitAntigen(dtRow);
				al.Add(bloodUnitAntigen);
			}
			return al;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1447"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5223"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.AntigenTypeId))
			{
				dtRow[TABLE.AntigenTypeId] = this.AntigenTypeId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitGuid))
			{
				dtRow[TABLE.BloodUnitGuid] = this.BloodUnitGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.Comments))
			{
				dtRow[TABLE.Comments] = this.Comments;
			}
			if(dtRow.Table.Columns.Contains(TABLE.DivisionCode))
			{
				dtRow[TABLE.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.LastUpdateDate))
			{
				dtRow[TABLE.LastUpdateDate] = this.LastUpdateDate;
			}
			if(dtRow.Table.Columns.Contains(TABLE.LastUpdateUser))
			{
				dtRow[TABLE.LastUpdateUser] = this.LastUpdateUser;
			}
			if(dtRow.Table.Columns.Contains(TABLE.Positive))
			{
				dtRow[TABLE.Positive] = this.Positive;
			}
			if(dtRow.Table.Columns.Contains(TABLE.RecordStatusCode))
			{
				dtRow[TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}
			if(dtRow.Table.Columns.Contains(TABLE.RowVersion))
			{
				if (this.IsNew) 
				{
					dtRow[TABLE.RowVersion] = DBNull.Value;
				}
				else
				{
					dtRow[TABLE.RowVersion] = this.RowVersion;
				}
			}
			return dtRow;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitGuid))
			{
				this.BloodUnitGuid = dtRow.IsNull(TABLE.BloodUnitGuid) ? System.Guid.Empty : (System.Guid) dtRow[TABLE.BloodUnitGuid];
			}
			if(dtRow.Table.Columns.Contains(TABLE.AntigenTypeId))
			{
				this.AntigenTypeId = dtRow.IsNull(TABLE.AntigenTypeId) ? System.Int32.MinValue : (System.Int32) dtRow[TABLE.AntigenTypeId];				
			}
			if(dtRow.Table.Columns.Contains(TABLE.Positive))
			{
				this.Positive = dtRow.IsNull(TABLE.Positive) ? false : System.Convert.ToBoolean(dtRow[TABLE.Positive]);				
			}
			if(dtRow.Table.Columns.Contains(Common.VbecsTables.AntigenType.AntigenTypeName))
			{
				this.AntigenName = dtRow.IsNull(Common.VbecsTables.AntigenType.AntigenTypeName) ? "" : dtRow[Common.VbecsTables.AntigenType.AntigenTypeName].ToString().Trim();				
			}
			if(dtRow.Table.Columns.Contains(TABLE.RecordStatusCode))
			{
				this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.RecordStatusCode].ToString());				
			}
			if(dtRow.Table.Columns.Contains(TABLE.RowVersion))
			{
				this.RowVersion = (byte[]) dtRow[TABLE.RowVersion];
			}
		}



		#endregion

	}
}


